<?php
/*
Plugin Name: ezdzsend
Description: Adds a download button to the thank you page and a custom “Downloads” tab in My Account.
Version: 2.0
Author: ezdzsend, info@ezdzsend.com
*/
if (!defined('ABSPATH')) exit;
/*
|--------------------------------------------------------------------------
| Enqueue CSS
|--------------------------------------------------------------------------
*/
add_action('admin_enqueue_scripts', function () {
    wp_enqueue_style('woocommerce_admin_styles', WC()->plugin_url() . '/assets/css/admin.css', [], WC_VERSION);
    wp_enqueue_style('ctd_admin_styles', plugins_url('assets/css/admin-style.css', __FILE__));
});
/*
|--------------------------------------------------------------------------
| Create DB Tables
|--------------------------------------------------------------------------
*/
register_activation_hook(__FILE__, 'ctd_install_tables');
function ctd_install_tables() {
    global $wpdb;
    $eligibility = $wpdb->prefix . 'ctd_product_eligibility';
    $downloads = $wpdb->prefix . 'ctd_user_download_new';
    $charset = $wpdb->get_charset_collate();
    $sql = "
        CREATE TABLE IF NOT EXISTS $eligibility (
            id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            store_id VARCHAR(255) NOT NULL,
            product_id BIGINT UNSIGNED NOT NULL,
            eligible BOOLEAN DEFAULT 0,
            UNIQUE KEY unique_product (store_id, product_id)
        ) $charset;
        CREATE TABLE IF NOT EXISTS $downloads (
            id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
            user_id BIGINT NOT NULL,
            product_id BIGINT NOT NULL,
            order_id VARCHAR(100) NOT NULL,
            download_url VARCHAR(500) NOT NULL,
            order_date DATETIME NOT NULL,
            download_count INT DEFAULT 0,
            PRIMARY KEY (id)
        ) $charset;
    ";
    require_once ABSPATH . 'wp-admin/includes/upgrade.php';
    dbDelta($sql);
    
    // Flush rewrite rules after adding endpoints
    ctd_add_downloads_endpoint();
    flush_rewrite_rules();
}

// Deactivation hook to clean up rewrite rules
register_deactivation_hook(__FILE__, 'ctd_deactivation');
function ctd_deactivation() {
    flush_rewrite_rules();
}
/*
|--------------------------------------------------------------------------
| Admin Menu + Settings Page
|--------------------------------------------------------------------------
*/
add_action('admin_menu', function () {
    add_submenu_page(
        'woocommerce',
        'Ezdzsend Settings',
        'Ezdzsend Eligibility',
        'manage_woocommerce',
        'download-eligibility',
        'ctd_render_admin_page'
    );
});
function ctd_render_admin_page() {
    global $wpdb;
    $table = $wpdb->prefix . 'ctd_product_eligibility';
    if ($_POST && check_admin_referer('ctd_save_settings')) {
        $store_id = sanitize_text_field($_POST['ctd_store_id']);
        update_option('ctd_store_id', $store_id);
        if (!empty($_POST['eligible'])) {
            foreach ($_POST['eligible'] as $pid => $v) {
                $wpdb->replace(
                    $table,
                    ['store_id' => $store_id, 'product_id' => (int)$pid, 'eligible' => $v === 'yes' ? 1 : 0],
                    ['%s', '%d', '%d']
                );
            }
        }
    }
    $store_id = get_option('ctd_store_id');
    ?>
    <div class="wrap woocommerce">
        <h1>Ezdzsend Product Eligibility</h1>
        <form method="post">
            <?php wp_nonce_field('ctd_save_settings'); ?>
            <table class="form-table">
                <tr>
                    <th>Store ID</th>
                    <td><input type="text" name="ctd_store_id" value="<?php echo esc_attr($store_id); ?>" class="regular-text"></td>
                </tr>
            </table>
            <h2>Eligible Products</h2>
            <table class="wp-list-table widefat striped">
                <thead>
                    <tr>
                        <th>S.No.</th>
                        <th>Product ID</th>
                        <th>Product Name</th>
                        <th>Eligible?</th>
                    </tr>
                </thead>
                <tbody>
                    <?php
                    $products = wc_get_products(['limit' => -1]);
                    $i = 1;
                    foreach ($products as $product) {
                        $pid = $product->get_id();
                        $name = $product->get_name();
                        $eligible = $wpdb->get_var(
                            $wpdb->prepare("SELECT eligible FROM $table WHERE store_id=%s AND product_id=%d", $store_id, $pid)
                        );
                        ?>
                        <tr>
                            <td><?php echo $i++; ?></td>
                            <td><?php echo $pid; ?></td>
                            <td><?php echo esc_html($name); ?></td>
                            <td>
                                <input type="hidden" name="eligible[<?php echo $pid; ?>]" value="no">
                                <input type="checkbox" name="eligible[<?php echo $pid; ?>]" value="yes" <?php checked($eligible, 1); ?>>
                            </td>
                        </tr>
                        <?php
                    }
                    ?>
                </tbody>
            </table>
            <p><input type="submit" class="button-primary" value="Save Changes"></p>
        </form>
    </div>
    <?php
}
/*
|--------------------------------------------------------------------------
| Thank You Page Download Button
|--------------------------------------------------------------------------
*/
add_action('woocommerce_thankyou', function ($order_id) {
    global $wpdb;
    $table = $wpdb->prefix . 'ctd_product_eligibility';
    $order = wc_get_order($order_id);
    if (!$order) return;
    $email = $order->get_billing_email();
    $order_number = $order->get_order_number();
    $store_id = get_option('ctd_store_id');
    $eligible = false;
    foreach ($order->get_items() as $item) {
        $pid = $item->get_product_id();
        if ($wpdb->get_var(
            $wpdb->prepare("SELECT eligible FROM $table WHERE store_id=%s AND product_id=%d", $store_id, $pid)
        )) {
            $eligible = true;
            break;
        }
    }
    if (!$eligible) return;
    $url = "https://app.ezdzsend.com/instant/download/?email=" . rawurlencode($email) .
           "&order_id=" . urlencode($order_number) .
           "&store_id=" . urlencode($store_id);
    ?>
    <div style="text-align:center; margin:20px 0;">
        <button onclick="window.location.href='<?php echo esc_url($url); ?>'"
                style="padding:14px 28px; background:black; color:white; border-radius:5px; border:none; cursor:pointer;">
            Download
        </button>
    </div>
    <?php
});
/*
|--------------------------------------------------------------------------
| Save Download History on Order Completion
|--------------------------------------------------------------------------
*/
add_action('woocommerce_order_status_completed', function ($order_id) {
    global $wpdb;
    $table = $wpdb->prefix . 'ctd_user_download_new';
    $order = wc_get_order($order_id);
    if (!$order) return;
    $user_id = $order->get_user_id();
    $store_id = get_option('ctd_store_id');
    $email = $order->get_billing_email();
    $order_number = $order->get_order_number();
    $date = $order->get_date_created()->date('Y-m-d H:i:s');
    $base_url = "https://app.ezdzsend.com/instant/download/?email=" . rawurlencode($email) .
                "&order_id=" . urlencode($order_number) .
                "&store_id=" . urlencode($store_id);
    foreach ($order->get_items() as $item) {
        $wpdb->insert($table, [
            'user_id' => $user_id,
            'product_id' => $item->get_product_id(),
            'order_id' => $order_number,
            'download_url' => esc_url_raw($base_url),
            'order_date' => $date
        ]);
    }
});
/*
|--------------------------------------------------------------------------
| Add Custom "Downloads" Tab
|--------------------------------------------------------------------------
*/
add_action('init', 'ctd_add_downloads_endpoint');
function ctd_add_downloads_endpoint() {
    add_rewrite_endpoint('custom-downloads', EP_ROOT | EP_PAGES);
}
add_filter('woocommerce_account_menu_items', function ($items) {
    $new = [];
    $i = 0;
    foreach ($items as $key => $val) {
        if ($i == 2) $new['custom-downloads'] = 'Custom Downloads';
        $new[$key] = $val;
        $i++;
    }
    return $new;
});
/*
|--------------------------------------------------------------------------
| Custom Downloads Page With Pagination
|--------------------------------------------------------------------------
*/
add_action('woocommerce_account_custom-downloads_endpoint', function () {
    global $wpdb, $current_user;
    $table = $wpdb->prefix . 'ctd_user_download_new';
    // Get the page number from the endpoint value for pretty URLs like /custom-downloads/2/
    $endpoint_val = get_query_var('custom-downloads');
    $page = ( ! empty( $endpoint_val ) && is_numeric( $endpoint_val ) ) ? absint( $endpoint_val ) : 1;
    $limit = 1;
    $offset = ( $page - 1 ) * $limit;
    $total = $wpdb->get_var(
        $wpdb->prepare("SELECT COUNT(*) FROM $table WHERE user_id=%d", $current_user->ID)
    );
    $rows = $wpdb->get_results(
        $wpdb->prepare("SELECT * FROM $table WHERE user_id=%d ORDER BY id DESC LIMIT %d OFFSET %d",
        $current_user->ID, $limit, $offset)
    );
    echo "<h2>Your Downloads</h2>";
    if ( empty( $rows ) ) {
        echo "<p>No downloads available.</p>";
        return;
    }
    echo '<table class="woocommerce-MyAccount-downloads">
            <thead><tr>
                <th>Product</th><th>Order ID</th><th>Date</th><th>Action</th>
            </tr></thead><tbody>';
    foreach ($rows as $d) {
        echo "<tr>
                <td>" . get_the_title( $d->product_id ) . "</td>
                <td>" . esc_html( $d->order_id ) . "</td>
                <td>" . date( 'F j, Y', strtotime( $d->order_date ) ) . "</td>
                <td><a class='button' href='" . esc_url( $d->download_url ) . "'>Download</a></td>
              </tr>";
    }
    echo '</tbody></table>';
    // Pagination using pretty URLs like /custom-downloads/2/
    $pages = ceil($total / $limit);

    if ($pages > 1) {

        echo '<nav class="woocommerce-pagination"><ul class="page-numbers">';

        $myaccount_page_id = get_option('woocommerce_myaccount_page_id');
        $myaccount_url = get_permalink($myaccount_page_id);

        $range = 2; // pages to show on each side
        $show_items = ($range * 2) + 1;

        // Previous link
        if ($page > 1) {
            $prev_url = wc_get_endpoint_url('custom-downloads', $page - 1, $myaccount_url);
            echo '<li><a class="prev page-numbers" href="' . esc_url($prev_url) . '">« Prev</a></li>';
        }

        // First page link
        if ($page > ($range + 1)) {
            $first_url = wc_get_endpoint_url('custom-downloads', 1, $myaccount_url);
            echo '<li><a class="page-numbers" href="' . esc_url($first_url) . '">1</a></li>';
            echo '<li><span class="page-numbers dots">…</span></li>';
        }

        // Main numbered links
        for ($i = max(1, $page - $range); $i <= min($pages, $page + $range); $i++) {
            if ($i == $page) {
                echo '<li><span class="page-numbers current">' . $i . '</span></li>';
            } else {
                $url = wc_get_endpoint_url('custom-downloads', $i, $myaccount_url);
                echo '<li><a class="page-numbers" href="' . esc_url($url) . '">' . $i . '</a></li>';
            }
        }

        // Last page link
        if ($page < ($pages - $range)) {
            echo '<li><span class="page-numbers dots">…</span></li>';
            $last_url = wc_get_endpoint_url('custom-downloads', $pages, $myaccount_url);
            echo '<li><a class="page-numbers" href="' . esc_url($last_url) . '">' . $pages . '</a></li>';
        }

        // Next link
        if ($page < $pages) {
            $next_url = wc_get_endpoint_url('custom-downloads', $page + 1, $myaccount_url);
            echo '<li><a class="next page-numbers" href="' . esc_url($next_url) . '">Next »</a></li>';
        }

        echo '</ul></nav>';

        // Pagination CSS
        echo '<style>
            .woocommerce-pagination .page-numbers {
                list-style: none;
                margin: 20px 0;
                padding: 0;
                display: flex;
                gap: 6px;
            }
            .woocommerce-pagination .page-numbers a,
            .woocommerce-pagination .page-numbers span {
                padding: 8px 14px;
                border: 1px solid #ddd;
                border-radius: 4px;
                text-decoration: none;
            }
            .woocommerce-pagination .page-numbers .current {
                background: #0073aa;
                color: #fff;
                border-color: #0073aa;
            }
            .woocommerce-pagination .page-numbers .dots {
                border: none;
            }
        </style>';
    }
});
/*
|--------------------------------------------------------------------------
| Disable Async Webhooks
|--------------------------------------------------------------------------
*/
add_filter('woocommerce_webhook_deliver_async', '__return_false');